
********************************************************************************
*                                                                              *
* 2 by 4 within subject design: Examples 9 and 10                               *
*                                                                              *
********************************************************************************

clear
set more off

use within2by4data 
browse

**********************
* Conventional ANOVA *
**********************

// Again we first conduct a conventional ANOVA to obtains SSmodel. To conduct 
// a conventional within-subject ANOVA, Stata wants the data to be in long 
// format:

rename noglasses_row1 retention1
rename noglasses_row2 retention2
rename noglasses_row3 retention3
rename noglasses_row4 retention4
rename glasses_row1 retention5
rename glasses_row2 retention6
rename glasses_row3 retention7
rename glasses_row4 retention8
reshape long retention, i(person) j(condition)  

gen sunglasses = 1 
replace sunglasses = 2 if condition > 4
gen location = 1 
replace location = 2 if condition == 2 | condition == 6
replace location = 3 if condition == 3 | condition == 7
replace location = 4 if condition == 4 | condition == 8

drop condition

//Next we run the repeated measures ANOVA

anova retention person sunglasses##location, repeated(sunglasses location) 

// As mentioned in Example 8, the SSmodel reported in the Stata output is  
// defined in ways different from the manuscript (see Equation 7). In the Stata  
// output SSmodel is the sum of SSsunglasses, SSlocation, SSsunglasses*location, 
// and SSperson. To obtain the requires SSmodel (i.e., the variance observed to 
// be related to the manipulation of sunglasses and seating location), the 
// SSperson has to be subtracted from the SSmodel as stored in Stata's memory as 
// "e(mss)". Since person is specified as the first factor in the ANOVA command
// above, SSperson is stored in Stata's memory as "e(ss_1)". Also, the 
// SSsunglasses*location*person in equation 10 is labeled SSresidual in the 
// Stata output table. 

// Let us obtain and store the required SSmodel in mata's memory so that it is 
// available when calculating ETA^2 altering later on. 

mata
SSmodel = st_numscalar("e(mss)")-st_numscalar("e(ss_1)")
SSmodel
end

**************************************
* Example 9: A direct test of theory *
**************************************

// Constrast analysis on repeated measures designs in STATA uses the 
// manovatest postestimation command of the MANOVA procedure. Since MANOVA 
// requires data to be in wide format, let us re-open the original
// dataset:

use within2by4data, clear

browse

// As explained in the manuscript, we typically can ignore the factorial
// structure of the experimental design, and treat the 2 by 4 design as a 1 by 
// 8. The analysis that is similar to repeated measures designs with a single
// within-subject factor (see Example 8). 

// As in Example 8, we first create a variable called "groups" which defines all
// participant as belonging to the same group:

gen groups = 1

// Next we run the MANOVA in exactly the same way as in Example 8. Note that 
// "///" means that the command continues on the next line:

manova noglasses_row1 noglasses_row2 noglasses_row3 noglasses_row4 ///
	glasses_row1 glasses_row2 glasses_row3 glasses_row4 = groups, noconstant

// Next, we define the required vector M with contrast weights:

matrix M = (3, 1, -1, -3/5, -3/5, -3/5, -3/5, -3/5)

// We use this vector M as input to the ytransform option of the manovatest
// postestimation command in exactly the same way as in Example 8:

manovatest groups, ytransform(M)

// Next, we store the SScontrast in mata's memory in exactly the same way as in 
// Example 8:

mata
SScontrast = st_matrix("r(H)")
end

// Next we can calculate the C-estimate and ETA^2 altering in exactly the same
// way as in Example 8:

means noglasses_row1 noglasses_row2 noglasses_row3 noglasses_row4 ///
	glasses_row1 glasses_row2 glasses_row3 glasses_row4

mata
Means = st_matrix("e(b)")
M = st_matrix("M")
Cest = rowsum(M :* Means) 
Cest
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting
end

**********************************************************
* Example 10: Negative linear effect of seating location *  
* without but not with sunglasses?                       *
**********************************************************

// As also mentioned in the annotated syntax of Example 6, Stata cannot test two
// separate theorectical predictions in one run. Instead the analysis has to be 
// repeated for each contrast separately. The required syntax is the same as in 
// Example 9, except for the constrast weights used. Note also that the "groups" 
// variable needed to run the MANOVA, the SSmodel, and the observed group means 
// (vector "Means") are already available from the analysis of Example 10: 

matrix M1 = (3, 1, -1, -3, 0, 0, 0, 0)
matrix M2 = (0, 0, 0, 0, 3, 1, -1, -3)

// Let us first test the prediction of which the contrast weights are specified
// in M1:

manova noglasses_row1 noglasses_row2 noglasses_row3 noglasses_row4 ///
	glasses_row1 glasses_row2 glasses_row3 glasses_row4 = groups, noconstant
manovatest groups, ytransform(M1)

mata
SScontrast = st_matrix("r(H)")
M = st_matrix("M1")
Cest = rowsum(M :* Means) 
Cest
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting
end

// Next we test the prediction of which the contrast weights are specified
// in M2:

manova noglasses_row1 noglasses_row2 noglasses_row3 noglasses_row4 ///
	glasses_row1 glasses_row2 glasses_row3 glasses_row4 = groups, noconstant
manovatest groups, ytransform(M2)

mata
SScontrast = st_matrix("r(H)")
M = st_matrix("M2")
Cest = rowsum(M :* Means) 
Cest
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting
end

// To obtain the multivariate test of the combined prediction as is reported by
// SPSS, the analysis needs to be repeated with, as input to the ytransform
// option, a matrix M that combines vectors M1 and M2 as rows (as in Example 4):

matrix M = (M1 \ M2)

manova noglasses_row1 noglasses_row2 noglasses_row3 noglasses_row4 ///
	glasses_row1 glasses_row2 glasses_row3 glasses_row4 = groups, noconstant
manovatest groups, ytransform(M)
